/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/


#ifndef _SCE_EVENT_H_
#define _SCE_EVENT_H_
namespace sce{ 
	namespace Toolkit{
		namespace NP{

	/// @brief
	/// Defines the different services provided by the %NP %Toolkit library.
	///
	/// Defines the different services provided by the %NP %Toolkit library. It is used
	/// by messages to define which service the message came from or is going to.
	/// It is also used by the <c>NpToolkitController</c> to ensure messages reach
	/// the correct service and by the <c>ServiceFactory</c> to define which services
	/// are being referred to.
	enum ServiceType {
		core	= 0,	///< Core services provided by the %NP %Toolkit library, which are not related to a particular service class.
		netInfo,		///< A service providing information about the network connection.
		profile	,		///< A service providing information about the current user's profile.
		friends,		///< A service managing friends lists and blocked lists etc.
		messaging,		///< A service sending messages to other PSN users.
		ranking,		///< A service managing scoreboards.
		trophy,			///< A service for managing trophies.
		presence,		///< A service providing status updates on the user's PSN profiles.
		tus,			///< A service for title user storage.
		tss,			///< A service for title small storage.
		matching,		///< A service providing matchmaking for online game play.
		sns,			///< A service for social networking services.
		auth,			///< A service used to retrieve a ticket from the PSN.
		commerce,		///< A service providing in-game commerce functionality.
		patching,		///< A service for checking availability of patches using the game update utility.
		wordFilter,		///< A service for censoring or sanitizing comments (or singular words).
		size			///< The number of services available.
	};

	/// @brief
	/// Represents an event generated by a service.
	///
	/// Event objects are created by the service and then passed through the
	/// <c>NpToolkitController</c>. From the <c>NpToolkitController</c> they are propagated to the application via a registered callback.
	/// This gives the external application important information on which events have been generated.
	struct Event {

		/// @brief
		/// Defines the different types of event that can be generated by the services.
		///
		/// Defines the different types of event that can be generated by the services.
		enum UserEvent 
		{
			unknown = -1,
			enetDown = 0,						///< An event from the NetCtl service generated when the connection layer has gone down.
			enetUp,								///< An event from the NetCtl service generated when a connection has been established.
			loggedIn,							///< An event from the NetCtl service generated when a connection to the PSN has been established.
			loggedOut,							///< An event from the NetCtl service generated when a connection to the PSN has been lost.

			netInfoGotBandwidth,				///< An event generated when the results from the bandwidth test have been obtained.
			netInfoGotBasic,					///< An event generated when basic network information has been obtained.
			netInfoGotDetailed,					///< An event generated when detailed network information has have been obtained.
			netInfoError,						///< An event generated when an error occurs in the net information service.
			netInfoDialogComplete,				///< An event generated when the log in dialog has been completed.

			profileGotOnlineId,					///< An event generated when the user's online ID has been retrieved.
			profileGotNpId,						///< An event generated when the user's %NP ID has been retrieved.
			profileGotOnlineName,				///< An event generated when the user's online name has been retrieved.
			profileGotAvatarUrl,				///< An event generated when the user's avatar URL has been retrieved.
			profileGotMyLanguages,				///< An event generated when the user's PSN languages have been retrieved.
			profileGotCachedUserInfo,			///< An event generated when a user's cached information has been retrieved.
			profileGotCountryInfo,				///< An event generated when the user's country details have been retrieved.
			profileGotParentalInfo,				///< An event generated when the user's parental control settings have been retrieved.
			profileGotPlatform,					///< An event generated when the current platform has been retrieved.

			friendsGotFriendsList,				///< An event generated when the user's friends list has been retrieved.
			friendsNoFriends,					///< An event generated if the user has no friends.
			friendsListReady,					///< An event generated when the user's friends list has been populated.
			friendsListUpdated,					///< An event generated when the user's friends list has been updated (a friend has been added/removed).
			friendsPresenceUpdated,				///< An event generated when the user's friends list has been updated with new presence information.
			friendsGuiExited,					///< An event generated when the GUI component for friends exits.
			blockListReady,						///< An event generated when the user's block list has been populated.
			blockListUpdated,					///< An event generated when the user's block list has been updated (a friend has been added/removed).


			messageAttachmentReceived,			///< An event generated when a message with a data attachment has been received.
			messageAttachmentOpened,			///< An event generated when a message with a data attachment has been opened (and the sysutil GUI is closed).
			messageInviteReceived,				///< An event generated when a message with an invite has been received.
			messageInviteAccepted,				///< An event generated when a message with an invite has been accepted via %NP %Toolkit (and the sysutil GUI is closed).
			messageSent,						///< An event generated when a message has been sent.

			presenceSet,						///< An event generated when presence has been set successfully.
			presenceSetFailed,					///< An event generated when presence set failed.

			rankingRangeRetrieved,					///< An event generated when a range of ranks from a scoreboard has been retrieved.
			rankingRangeRetrievedFail,				///< An event generated when a range of ranks from a scoreboard failed to be retrieved.
			rankingFriendsRetrieved,				///< An event generated when the ranks and scores belonging a user's friend have been retrieved.
			rankingFriendsRetrievedFail,			///< An event generated when the rank of a user's friend could not be retrieved.
			rankingFriendsRetrievedFailNoFriends,	///< An event generated when a user has no friends so ranking service cannot perform a request.
			rankingMatchingBoardFound,				///< An event generated when a board matching the arguments was found.
			rankingHighScore,						///< An event generated when a higher score was already recorded in the cache.
			rankingServerError,						///< An event generated when an error occurred with the ranking server.
			rankingCommunityError,					///< An event generated when an error occurred communicating with the ranking server.
			rankingScoreRegistered,					///< An event generated when a ranking score has been registered.
			rankingScoreRegisteredFail,				///< An event generated when an attempt to register a ranking score has failed.
			rankingScoreRegisteredFailNotBest,		///< An event generated when an attempt to register a ranking score has failed.
			rankingMemoryError,						///< An event generated when the ranking service cannot allocate anymore memory for the cache.	
			rankingUserRankRetrieved,				///< An event generated when the user rank has been retrieved.
			rankingUserRankRetrievedFail,			///< An event generated when the user rank has failed to be retrieved.
			rankingInitFail,						///< An event generated when the ranking initialization failed.	
			rankingNotOnline,						///< An event generated when the ranking server is not online.

			matchingSessionCreated,				///< An event generated when session creation has been completed.
			matchingSessionJoined,				///< An event generated when the join session process has been completed.
			matchingSessionSearchCompleted,		///< An event generated when the search process has been completed.
			matchingSessionError,				///< An event generated when there was error performing the current process.
			matchingSessionLeft,				///< An event generated when the user has left the current session.
			matchingSessionModified,			///< An event generated when the session has been modified.
			matchingSessionUpdate,				///< An event generated when the session has been updated.
			matchingSessionMessageSentError,	///< An event generated when the session message has been sent.	

			commerceSessionCreated,				///< An event generated when a commerce session has successfully been created.
			commerceSessionAborted,				///< An event generated when the creation of commerce session has been aborted.
			commerceGotCategoryInfo,			///< An event generated when some category information has been retrieved from the store.
			commerceGotProductList,				///< An event generated when a list of products that are available has been retrieved from the store.
			commerceGotDetailedProductInfo,		///< An event generated when some detailed product information has been retrieved from the store.
			commerceProductBrowseStarted,		///< An event generated when product overlay has started.
			commerceProductBrowseSuccess,		///< An event generated when a product browse was completed successfully, and the user purchased the product.
			commerceProductBrowseAborted,		///< An event generated when a product browse was aborted by the user (the user pressed back).
			commerceProductBrowseFinished,		///< An event generated when a product browse has finished and it is now safe to free memory.
			commerceVoucherInputStarted,		///< An event generated when a voucher code input overlay was started.
			commerceVoucherInputSuccess,		///< An event generated when a voucher code input completed successfully.
			commerceVoucherInputAborted,		///< An event generated when a voucher code input was aborted by the user (user pressed back).
			commerceVoucherInputFinished,		///< An event generated when a voucher code input has finished. It is now safe to free memory.
			commerceGotEntitlementList,			///< An event generated when a the list of entitlements has been received for the current user.
			commerceConsumedEntitlement,		///< An event generated when the has successfully consumed an entitlement.
			commerceCheckoutStarted,			///< An event generated when a store checkout overlay has started.
			commerceCheckoutSuccess,			///< An event generated when user has successfully purchased from the checkout.
			commerceCheckoutAborted,			///< An event generated when the checkout was aborted by the user (user pressed back).
			commerceCheckoutFinished,			///< An event generated when a store checkout overlay has finished.
			commerceDownloadListStarted,		///< An event generated when a download list overlay has started.
			commerceDownloadListSuccess,		///< An event generated when the user has ended the download list.
			commerceDownloadListFinished,		///< An event generated when a download list overlay has finished.
			commerceError,						///< An event generated when a commerce error has occurred.

			snsMessagePosted,					///< An event generated when a message was successfully posted to Facebook.
			snsError,							///< An event generated when a sns specific error occurred.	

			authGotTicket,						///< An event generated when a ticket from PSN has been received.
			authNewTicket,						///< An event generated when a new ticket is available.
			authError,							///< An event generated when an authentication error occurred.

			trophyNotInit,						///< An event generated when the trophy service is not initialized.
			trophyContextFail,					///< An event generated when there is a context error for the trophy service.
			trophyNotEnoughSpace,				///< An event generated when there is not enough space returned from trying to register trophy set.
			trophyListRetrievalSuccess,			///< An event generated when a trophy list was retrieved successfully.
			trophyListRetrievalFail,			///< An event generated when a trophy list failed to be retrieved.
			trophyPlatinumUnlocked,				///< An event generated when a platinum trophy is unlocked.
			trophyUnlockSuccess,				///< An event generated when a trophy was unlocked successfully.
			trophyUnlockFail,					///< An event generated when a trophy could not be unlocked.
			trophyAlreadyUnlocked,				///< An event generated when a trophy is already unlocked.
			trophyAlreadyRegistered,			///< An event generated when a trophy set is already registered.	
			trophyInvalidID,					///< An event generated when an invalid trophy ID was passed to the trophy service.
			trophySetSetupSuccess,				///< An event generated when a trophy set setup was successful.
			trophySetSetupFail,					///< An event generated when a trophy set setup failed.
			trophyListAlreadyRetrieving,		///< An event generated when the trophy service is already in the process of trying to retrieve another list.
			trophyGameInfoRetrievalSuccess,		///< An event generated when group information retrieval was successful.
			trophyGameInfoRetrievalFail,		///< An event generated when group information retrieval failed.
			trophyBusy,							///< An event generated when the trophy service is busy and therefore cannot process a request.
			trophyTerminated,					///< An event generated when the trophy service has been terminated.
			trophyProgressSuccess,				///< An event generated when the progress of the user has been retrieved.
			trophyProgressFail,					///< An event generated when the progress of the user failed to be retrieved.
			trophyCacheReady,					///< An event generated when the trophy cache is ready to use.	

		patchingError,						///< An event generated when a game update utility error occurred.
		patchingChecked,					///< An event generated when the game update utility succeeded in checking for a patch.

		tssGotData,							///< An event generated when data has been retrieved from a TSS (title small storage) server.
		tssNoData,							///< An event generated when no data is found on a TSS (title small storage) server.
		tssError,							///< An event generated when an error occurs while working with a TSS (title small storage) server.

		tusDataSet,							///< An event generated when data is uploaded to a TUS (title user storage) server.
		tusDataReceived,					///< An event generated when data is retrieved from a TUS (title user storage) server.
		tusVariablesSet,					///< An event generated when variables are set on a TUS (title user storage) server.
		tusVariablesReceived,				///< An event generated when variables are retrieved from a TUS (title user storage) server.
		tusCrossSaveDataSet,				///< An event generated when data is uploaded to a TUS (title user storage) server for cross-saves.
		tusCrossSaveDataReceived,			///< An event generated when data is retrieved from a TUS (title user storage) server for cross-saves.
		tusError,							///< An event generated when an error occurs while working with a TUS (title user storage) server.

		wordFilterComplete,					///< An event generated when word filter operation has completed.

	    serviceTerminate,					///< The event message received by application when request to terminate service is processed
		serviceError						///< A generic error event relate to the service sepcified in the service parameter of <c>Event</c> structure.	
};

		/// @brief
		/// The default constructor for the Event struct.
		///
		/// The default constructor for the Event struct.
		Event() : service(core), event(unknown), returnCode(0){}

		ServiceType		service;	///< The Service this Event was generated by.
		UserEvent		event;		///< The event that was generated by the Service.
		int				returnCode;	///< The return code of the Event if applicable.
	};

	/// @brief
	/// Representing a callback event generated for the services.
	///
	/// Representing a callback event generated for the services. Callback Event objects are created by the SysutilCallbacks. These execute in the main
	/// thread context so have to quickly package the details into a CallbackEvent and
	/// pass it onto the <c>Thread</c> queue for processing by the relevant services.
	struct CallbackEvent 
	{
		/// @brief
		/// The default constructor.
		///
		/// The default constructor. Sets the basic type to <c>sysUtil</c> and the priority to 10. Everything else is initialized to 0.
		CallbackEvent() :	type(sysUtil),
							event(0),
							result(0),
							arg1(0),
							args(),
							priority(10) {}
		/// @brief
		/// Defines the type of callback that was triggered. 
		///
		/// Defines the type of callback that was triggered. This maps to the actual %NP libraries rather than services.
		enum CallbackType {
			sysUtil =0, ///< %NP %Toolkit SysUtil Callback triggered.
			netCtl	=1,	///< A Net Ctl library for monitoring the ethernet connection.
			manager	=2, ///< A manager library for ticketing, profiles and user information.
			basic	=3, ///< A basic library for presence, messaging and friends.
			trophy	=4, ///< A trophy library.
			ranking	=5,	///< A ranking library
			matching =6, ///< A matching library.
			wordFilter =7 /// A word filter library.
		};

		CallbackType		type;		///< The underlying callback this event was generated by.
		long int			event;		///< The event as an integer to indicate which event. Callbacks don't typically return much more data.
		long int			result;		///< An optional result code.
		unsigned long int	arg1;		///< An optional extra argument to avoid memory allocation if it will fit in 64 bits.
		void				*args;		///< Optional extra data.
		unsigned short		priority;	///< The priority of the message. Messages are processed in order of priority. The default is 10.

		/// @brief
		/// Lesser than operator which compares the priority of this CallbackEvent object with another.
		///
		/// Lesser than operator which compares the priority of this CallbackEvent object with another.
		/// @param rhs The CallbackEvent object to compare this one against.
		/// @return
		/// Returns true if the supplied CallbackEvent object has a lower priority than this one. False is returned if it has the same or a greater priority.
		bool operator<(CallbackEvent rhs) const
		{
			if(this->priority == rhs.priority) //if same priority say the first inserted is higher priority (lower number)
				return false;
			return this->priority > rhs.priority; // lowest number priority has highest precedence.
		}
	};

		}//end NP
	}//end Toolkit 
}//end Sce

#endif
